## ----setup, include=FALSE--------------------------------------------------
knitr::opts_chunk$set(echo = FALSE, warning = FALSE, message = FALSE)
library(causaloptim)
library(shiny)
library(kableExtra)

latex <- function(html = TRUE) {
  
  if(html) {
    ('LaTeX')
  } else {
    ('\\LaTeX')
  } 
  
}


## ----InterfaceStart, out.width = "100%", fig.cap = "The Shiny web interface at lauch", fig.alt = "A picture of the causaloptim Shiny app web interface.", fig.align = 'center', fig.show = 'hold'----
knitr::include_graphics("InterfaceStart.png")


## ----DAG, echo=FALSE, fig.align='default', out.width = '50%', fig.show='hold', fig.cap="Constructing the DAG", fig.subcap=c("Adding and naming variables", "Adding directed edges")----
knitr::include_graphics(c("InterfaceAddNode.png", "InterfaceAddedArcs.png"))


## ----Cardinality, echo=FALSE, fig.align='default', out.width = '33%', fig.show='hold', fig.cap="Setting attributes", fig.subcap=c("Setting the number of categories", "Confirmation message", "Setting exposure and outcome")----
knitr::include_graphics(c("InterfaceSetCard.png", "InterfaceCardSet.png", "InterfaceAttribSet.png"))


## ----causalDAG, echo=FALSE, fig.align='default', out.width = '50%', fig.show='hold', fig.cap="The causal DAG and bounds", fig.subcap=c("Graphical summary of the DAG with added confounding", "Computing the bounds")----
knitr::include_graphics(c("InterfaceQuery.png", "InterfaceBounds.png"))


## ---- include = knitr::is_html_output()------------------------------------
fluidPage(
    headerPanel("Causal Network Analysis and Optimization"), 
    fluidRow(id = "helptext", 
             column(12, helpText("The graph is divided into a left side and a right side. There is no confounding allowed between the left and right sides. All variables on the right side are confounded. Connections between the left and right sides must originate from the left. I.e, no connections from the right to left are allowed. On the left side, arbitrary connections are allowed and the model assumes that all nodes are observed and connected. On the right side, unmeasured variables are allowed, and the procedure assumes unmeasured confounding between all nodes. Once you press 'Analyze the graph', the algorithm will automatically add common causes to each side. ")), 
             column(12, helpText("Shift+click to add nodes. Shift+drag to connect nodes. Click to select nodes/edges and press 'd' to remove. Click a node to select and then press 'u' to mark it as unobserved/latent or press 'y' to mark it as the outcome of interest, or press 'e' to mark it as the exposure of interest. Select a node and press a digit to set that number of possible categorical values (all variables default to binary), or press 'c' and enter a number into the prompt. Click an edge and press 'm' to enforce monotonicity for that connection. Other constraints can be specified later. "))
             ),
    fluidRow(id = "graphrow",
        column(12, 
               tagList(tags$div(id="nvalsModal", class="modal fade", role="dialog", 
                                tags$div(class="modal-dialog modal-dialog-centered", role="document", 
                                         tags$div(class="modal-content shadow", 
                                                  tags$div(class="modal-header", 
                                                           tags$h5(class="modal-title", "Enter number of values: ")
                                                           ), 
                                                  tags$div(class="modal-body", 
                                                           tags$input(id="nvalsInput", type="number", class="form-control", value=2, min=2)
                                                           ), 
                                                  tags$div(class="modal-footer", 
                                                           tags$button(type="button", class="btn btn-default", `data-dismiss`="modal", "Cancel"), 
                                                           tags$button(id="nvalsSubmit", type="button", class="btn btn-primary", `data-toggle`="modal", `data-target`="#nvalsModal", "Ok")))))), 
               tagList(tags$div(id="toast")), 
               tagList(tags$div(id="graph", style="height:480px"), 
                          tags$br()))
    ), 
    #verbatimTextOutput("outcode"), # for debugging
    actionButton("endbtn", "Exit and return graph object", style="background-color: #fb6970"),
    actionButton("analyze", "Analyze the graph", style="background-color: #69fb82"), 
    
    tags$script(src = "d3.v3.min.js"),
    tags$script(src = "jquery.js"),
    tags$script(src = "shiny.js"),
    tags$script(src = "graph-creator.js")
)


## ---- echo = TRUE, collapse = TRUE-----------------------------------------
graph <- igraph::graph_from_literal(Z -+ X, X -+ Y, 
                                    Ul -+ Z, Ur -+ X, Ur -+ Y)
V(graph)$leftside <- c(1, 0, 0, 1, 0)
V(graph)$latent   <- c(0, 0, 0, 1, 1)
V(graph)$nvals    <- c(2, 2, 2, 2, 2)
E(graph)$rlconnect     <- c(0, 0, 0, 0, 0)
E(graph)$edge.monotone <- c(0, 0, 0, 0, 0)

riskdiff <- "p{Y(X = 1) = 1} - p{Y(X = 0) = 1}"
obj <- analyze_graph(graph, constraints = NULL, effectt = riskdiff)
bounds <- optimize_effect_2(obj)
bounds


## ---- echo = TRUE----------------------------------------------------------
bounds_function <- interpret_bounds(bounds$bounds, obj$parameters)
str(bounds_function)


## ----Overview, echo=FALSE, fig.align='center', out.width = '100%', fig.show='hold', fig.cap="Function Overview Flow Chart"----
knitr::include_graphics("causaloptimDiagram.drawio.png")


## \begin{tikzpicture}

## \definecolor{rightcol}{HTML}{FFFF00}

## \definecolor{leftcol}{HTML}{0000FF}

## \draw[fill=rightcol, opacity=0.1, very thick] (3,-1) -- (3,3) -- (7,3) -- (7,-1) -- (3,-1);

## \draw[fill=leftcol, opacity=0.1, very thick] (-1,-1) -- (3,-1) -- (3,3) -- (-1,3) -- (-1,-1);

## \draw[opacity=0.9, thick] (-1,-1) -- (3,-1) -- (3,3) -- (-1,3) -- (-1,-1);

## \draw[opacity=0.9, very thick] (3,-1) -- (3,3) -- (7,3) -- (7,-1) -- (3,-1);

## \node[draw, circle] at (1, 0) (X) {$X$};

## \node[draw, circle] at (4, 1) (M) {$M$};

## \node[draw, circle] at (6, 0) (Y) {$Y$};

## \node[draw, circle, dashed] at (5, 2) (Ur) {$U_r$};

## \node[draw, circle, dashed] at (1, 2) (Ul) {$U_l$};

## \draw [->] (X) to (M);

## \draw [->] (X) to (Y);

## \draw [->] (M) to (Y);

## \draw [->] (Ur) to (M);

## \draw [->] (Ur) to (Y);

## \draw [->] (Ul) to (X);

## \end{tikzpicture}


## ----mediation-dag, echo = TRUE--------------------------------------------
b <- igraph::graph_from_literal(X -+ Y, X -+ M, M -+ Y, 
                                Ul -+ X, Ur -+ Y, Ur -+ M)
V(b)$leftside <- c(1, 0, 0, 1, 0)
V(b)$latent   <- c(0, 0, 0, 1, 1)
V(b)$nvals    <- c(2, 2, 2, 2, 2)
E(b)$rlconnect     <- c(0, 0, 0, 0, 0, 0)
E(b)$edge.monotone <- c(0, 0, 0, 0, 0, 0)


## ----mediation-data, echo = TRUE-------------------------------------------
# parameters of the form pab_c, which represents 
# the probability P(Y = a, M = b | X = c)
p00_0 <- 1426/1888 # P(Y=0,M=0|X=0)
p10_0 <- 97/1888 # P(Y=1,M=0|X=0)
p01_0 <- 332/1888 # P(Y=0,M=1|X=0)
p11_0 <- 33/1888 # P(Y=1,M=1|X=0)
p00_1 <- 1081/1918 # P(Y=0,M=0|X=1)
p10_1 <- 86/1918 # P(Y=1,M=0|X=1)
p01_1 <- 669/1918 # P(Y=0,M=1|X=1)
p11_1 <- 82/1918 # P(Y=1,M=1|X=1)


## ----mediation-computation, echo = TRUE------------------------------------
CDE0_query <- "p{Y(M = 0, X = 1) = 1} - p{Y(M = 0, X = 0) = 1}"
CDE0_obj <- analyze_graph(b, constraints = NULL, effectt = CDE0_query)
CDE0_bounds <- optimize_effect_2(CDE0_obj)
CDE0_boundsfunction <- interpret_bounds(bounds = CDE0_bounds$bounds, 
                                        parameters = CDE0_obj$parameters)
CDE0_numericbounds <- CDE0_boundsfunction(p00_0 = p00_0, p00_1 = p00_1, 
                                          p10_0 = p10_0, p10_1 = p10_1, 
                                          p01_0 = p01_0, p01_1 = p01_1, 
                                          p11_0 = p11_0, p11_1 = p11_1)

CDE1_query <- "p{Y(M = 1, X = 1) = 1} - p{Y(M = 1, X = 0) = 1}"
CDE1_obj <- update_effect(CDE0_obj, effectt = CDE1_query)
CDE1_bounds <- optimize_effect_2(CDE1_obj)
CDE1_boundsfunction <- interpret_bounds(bounds = CDE1_bounds$bounds, 
                                        parameters = CDE1_obj$parameters)
CDE1_numericbounds <- CDE1_boundsfunction(p00_0 = p00_0, p00_1 = p00_1, 
                                          p10_0 = p10_0, p10_1 = p10_1, 
                                          p01_0 = p01_0, p01_1 = p01_1, 
                                          p11_0 = p11_0, p11_1 = p11_1)
NDE0_query <- "p{Y(M(X = 0), X = 1) = 1} - p{Y(M(X = 0), X = 0) = 1}"
NDE0_obj <- update_effect(CDE0_obj, effectt = NDE0_query)
NDE0_bounds <- optimize_effect_2(NDE0_obj)
NDE0_boundsfunction <- interpret_bounds(bounds = NDE0_bounds$bounds, 
                                        parameters = NDE0_obj$parameters)
NDE0_numericbounds <- NDE0_boundsfunction(p00_0 = p00_0, p00_1 = p00_1, 
                                          p10_0 = p10_0, p10_1 = p10_1, 
                                          p01_0 = p01_0, p01_1 = p01_1, 
                                          p11_0 = p11_0, p11_1 = p11_1)

NDE1_query <- "p{Y(M(X = 1), X = 1) = 1} - p{Y(M(X = 1), X = 0) = 1}"
NDE1_obj <- update_effect(CDE0_obj, effectt = NDE1_query)
NDE1_bounds <- optimize_effect_2(NDE1_obj)
NDE1_boundsfunction <- interpret_bounds(bounds = NDE1_bounds$bounds, 
                                        parameters = NDE1_obj$parameters)
NDE1_numericbounds <- NDE1_boundsfunction(p00_0 = p00_0, p00_1 = p00_1, 
                                          p10_0 = p10_0, p10_1 = p10_1, 
                                          p01_0 = p01_0, p01_1 = p01_1, 
                                          p11_0 = p11_0, p11_1 = p11_1)


## ----mediation-bounds------------------------------------------------------
# mediation effects
numeric_bounds <- rbind("CDE(0)" = CDE0_numericbounds, 
                        "CDE(1)" = CDE1_numericbounds, 
                        "NDE(0)" = NDE0_numericbounds, 
                        "NDE(1)" = NDE1_numericbounds)
knitr::kable(numeric_bounds, digits = 2, caption = "Bounds on the controlled and natural direct effects.")


## ----mendelian-data, echo = TRUE-------------------------------------------
params <- list(p00_0 = 0.83, p00_1 = 0.88, p00_2 = 0.72, 
               p10_0 = 0.11, p10_1 = 0.05, p10_2 = 0.20, 
               p01_0 = 0.05, p01_1 = 0.06, p01_2 = 0.05, 
               p11_0 = 0.01, p11_1 = 0.01, p11_2 = 0.03)


## ----causaloptim-computation, echo = TRUE----------------------------------
# Input causal DAG
b <- graph_from_literal(Z -+ X, Ul -+ Z, X -+ Y, Ur -+ X, Ur -+ Y)
V(b)$leftside <- c(1, 0, 1, 0, 0)
V(b)$latent <- c(0, 0, 1, 0, 1)
V(b)$nvals <- c(3, 2, 2, 2, 2)
E(b)$rlconnect <- c(0, 0,  0, 0, 0)
E(b)$edge.monotone <- c(0, 0, 0, 0, 0)
# Construct causal problem
obj <- analyze_graph(b, constraints = NULL, 
                     effectt = "p{Y(X = 1) = 1} - p{Y(X = 0) = 1}")
# Compute bounds on query
bounds <- optimize_effect_2(obj)
# Construct bounds as function of parameters
boundsfunction <- interpret_bounds(bounds = bounds$bounds, 
                                   parameters = obj$parameters)
# Insert observed conditional probabilities
numericbounds <- do.call(boundsfunction, as.list(params))
round(numericbounds, 2)

